/*************************************************************************
**       Advanced Encryption Standard - 128 - implementation en C       **
**       Chiffrement, dechiffrement, cration de clef                   **
**       Par Lillo Adrien et Comte-Gaz Quentin                          **
*************************************************************************/

#include<stdio.h>
#include<stdlib.h>
#include<time.h>
#include<string.h>
#define SUM4(s) (((s) &  1) ^ (((s) & 2) >> 1)  ^ (((s) & 4) >> 2)  ^ (((s) & 8) >> 3))

// Macro de multiplication par x 	
#define xtime(x)   ((((x)<<1) ^ ((((x)>>7) & 1) * 27)) & 255)


/*------------------------ Les tables et les constantes -------------------*/

#define Nk 4
int TAILLEMESSAGE=0; // On initialise la taille du message  0
int TEST=0; // Cette variable permet de passer en phase de test de norme

typedef unsigned int UINT;

// Les blocks de  messages sont stocks sur un tableau de 16 uchar
// M=[ M_0 .. M_3 | M_4 .. M_7 | M_8 .. M_11 | M_12 .. M_15 ] 
// M=[ colonne 0  | colonne 1  | colonne 2   | colonne 3    ]





// Le tableau de la fonction de substitution
int sbox[256] =   {
    0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
	0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
	0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
	0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
	0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
	0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
	0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
	0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
	0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
	0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
	0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
	0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
	0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
	0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
	0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
	0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16 };

int sboxinv[256] ={
    0x52 ,0x09 ,0x6a ,0xd5 ,0x30 ,0x36 ,0xa5 ,0x38 ,0xbf ,0x40 ,0xa3 ,0x9e ,0x81 ,0xf3 ,0xd7 ,0xfb
    ,0x7c ,0xe3 ,0x39 ,0x82 ,0x9b ,0x2f ,0xff ,0x87 ,0x34 ,0x8e ,0x43 ,0x44 ,0xc4 ,0xde ,0xe9 ,0xcb
    ,0x54 ,0x7b ,0x94 ,0x32 ,0xa6 ,0xc2 ,0x23 ,0x3d ,0xee ,0x4c ,0x95 ,0x0b ,0x42 ,0xfa ,0xc3 ,0x4e
    ,0x08 ,0x2e ,0xa1 ,0x66 ,0x28 ,0xd9 ,0x24 ,0xb2 ,0x76 ,0x5b ,0xa2 ,0x49 ,0x6d ,0x8b ,0xd1 ,0x25
    ,0x72 ,0xf8 ,0xf6 ,0x64 ,0x86 ,0x68 ,0x98 ,0x16 ,0xd4 ,0xa4 ,0x5c ,0xcc ,0x5d ,0x65 ,0xb6 ,0x92
    ,0x6c ,0x70 ,0x48 ,0x50 ,0xfd ,0xed ,0xb9 ,0xda ,0x5e ,0x15 ,0x46 ,0x57 ,0xa7 ,0x8d ,0x9d ,0x84
    ,0x90 ,0xd8 ,0xab ,0x00 ,0x8c ,0xbc ,0xd3 ,0x0a ,0xf7 ,0xe4 ,0x58 ,0x05 ,0xb8 ,0xb3 ,0x45 ,0x06
    ,0xd0 ,0x2c ,0x1e ,0x8f ,0xca ,0x3f ,0x0f ,0x02 ,0xc1 ,0xaf ,0xbd ,0x03 ,0x01 ,0x13 ,0x8a ,0x6b
    ,0x3a ,0x91 ,0x11 ,0x41 ,0x4f ,0x67 ,0xdc ,0xea ,0x97 ,0xf2 ,0xcf ,0xce ,0xf0 ,0xb4 ,0xe6 ,0x73
    ,0x96 ,0xac ,0x74 ,0x22 ,0xe7 ,0xad ,0x35 ,0x85 ,0xe2 ,0xf9 ,0x37 ,0xe8 ,0x1c ,0x75 ,0xdf ,0x6e
    ,0x47 ,0xf1 ,0x1a ,0x71 ,0x1d ,0x29 ,0xc5 ,0x89 ,0x6f ,0xb7 ,0x62 ,0x0e ,0xaa ,0x18 ,0xbe ,0x1b
    ,0xfc ,0x56 ,0x3e ,0x4b ,0xc6 ,0xd2 ,0x79 ,0x20 ,0x9a ,0xdb ,0xc0 ,0xfe ,0x78 ,0xcd ,0x5a ,0xf4
    ,0x1f ,0xdd ,0xa8 ,0x33 ,0x88 ,0x07 ,0xc7 ,0x31 ,0xb1 ,0x12 ,0x10 ,0x59 ,0x27 ,0x80 ,0xec ,0x5f
    ,0x60 ,0x51 ,0x7f ,0xa9 ,0x19 ,0xb5 ,0x4a ,0x0d ,0x2d ,0xe5 ,0x7a ,0x9f ,0x93 ,0xc9 ,0x9c ,0xef
    ,0xa0 ,0xe0 ,0x3b ,0x4d ,0xae ,0x2a ,0xf5 ,0xb0 ,0xc8 ,0xeb ,0xbb ,0x3c ,0x83 ,0x53 ,0x99 ,0x61
    ,0x17 ,0x2b ,0x04 ,0x7e ,0xba ,0x77 ,0xd6 ,0x26 ,0xe1 ,0x69 ,0x14 ,0x63 ,0x55 ,0x21 ,0x0c ,0x7d};



// Rijndael key schedule
//http://en.wikipedia.org/wiki/Rijndael_key_schedule
//But : Expand a short key into a number of separate round keys.
int Rcon[255] = {
	0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 
	0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 
	0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 
	0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 
	0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 
	0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 
	0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 
	0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 
	0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 
	0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 
	0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 
	0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 
	0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 
	0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 
	0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 
	0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb  };

// Table de multiplication  utiliser dans InvMixColumns (tables de Galois(2^8)
//http://en.wikipedia.org/wiki/Rijndael_mix_columns 

int Mul2[256] = {
    0x00,0x02,0x04,0x06,0x08,0x0a,0x0c,0x0e,0x10,0x12,0x14,0x16,0x18,0x1a,0x1c,0x1e,
    0x20,0x22,0x24,0x26,0x28,0x2a,0x2c,0x2e,0x30,0x32,0x34,0x36,0x38,0x3a,0x3c,0x3e,
    0x40,0x42,0x44,0x46,0x48,0x4a,0x4c,0x4e,0x50,0x52,0x54,0x56,0x58,0x5a,0x5c,0x5e,
    0x60,0x62,0x64,0x66,0x68,0x6a,0x6c,0x6e,0x70,0x72,0x74,0x76,0x78,0x7a,0x7c,0x7e,
    0x80,0x82,0x84,0x86,0x88,0x8a,0x8c,0x8e,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,
    0xa0,0xa2,0xa4,0xa6,0xa8,0xaa,0xac,0xae,0xb0,0xb2,0xb4,0xb6,0xb8,0xba,0xbc,0xbe,
    0xc0,0xc2,0xc4,0xc6,0xc8,0xca,0xcc,0xce,0xd0,0xd2,0xd4,0xd6,0xd8,0xda,0xdc,0xde,
    0xe0,0xe2,0xe4,0xe6,0xe8,0xea,0xec,0xee,0xf0,0xf2,0xf4,0xf6,0xf8,0xfa,0xfc,0xfe,
    0x1b,0x19,0x1f,0x1d,0x13,0x11,0x17,0x15,0x0b,0x09,0x0f,0x0d,0x03,0x01,0x07,0x05,
    0x3b,0x39,0x3f,0x3d,0x33,0x31,0x37,0x35,0x2b,0x29,0x2f,0x2d,0x23,0x21,0x27,0x25,
    0x5b,0x59,0x5f,0x5d,0x53,0x51,0x57,0x55,0x4b,0x49,0x4f,0x4d,0x43,0x41,0x47,0x45,
    0x7b,0x79,0x7f,0x7d,0x73,0x71,0x77,0x75,0x6b,0x69,0x6f,0x6d,0x63,0x61,0x67,0x65,
    0x9b,0x99,0x9f,0x9d,0x93,0x91,0x97,0x95,0x8b,0x89,0x8f,0x8d,0x83,0x81,0x87,0x85,
    0xbb,0xb9,0xbf,0xbd,0xb3,0xb1,0xb7,0xb5,0xab,0xa9,0xaf,0xad,0xa3,0xa1,0xa7,0xa5,
    0xdb,0xd9,0xdf,0xdd,0xd3,0xd1,0xd7,0xd5,0xcb,0xc9,0xcf,0xcd,0xc3,0xc1,0xc7,0xc5,
    0xfb,0xf9,0xff,0xfd,0xf3,0xf1,0xf7,0xf5,0xeb,0xe9,0xef,0xed,0xe3,0xe1,0xe7,0xe5};

int Mul3[256] = {
    0x00,0x03,0x06,0x05,0x0c,0x0f,0x0a,0x09,0x18,0x1b,0x1e,0x1d,0x14,0x17,0x12,0x11,
    0x30,0x33,0x36,0x35,0x3c,0x3f,0x3a,0x39,0x28,0x2b,0x2e,0x2d,0x24,0x27,0x22,0x21,
    0x60,0x63,0x66,0x65,0x6c,0x6f,0x6a,0x69,0x78,0x7b,0x7e,0x7d,0x74,0x77,0x72,0x71,
    0x50,0x53,0x56,0x55,0x5c,0x5f,0x5a,0x59,0x48,0x4b,0x4e,0x4d,0x44,0x47,0x42,0x41,
    0xc0,0xc3,0xc6,0xc5,0xcc,0xcf,0xca,0xc9,0xd8,0xdb,0xde,0xdd,0xd4,0xd7,0xd2,0xd1,
    0xf0,0xf3,0xf6,0xf5,0xfc,0xff,0xfa,0xf9,0xe8,0xeb,0xee,0xed,0xe4,0xe7,0xe2,0xe1,
    0xa0,0xa3,0xa6,0xa5,0xac,0xaf,0xaa,0xa9,0xb8,0xbb,0xbe,0xbd,0xb4,0xb7,0xb2,0xb1,
    0x90,0x93,0x96,0x95,0x9c,0x9f,0x9a,0x99,0x88,0x8b,0x8e,0x8d,0x84,0x87,0x82,0x81,
    0x9b,0x98,0x9d,0x9e,0x97,0x94,0x91,0x92,0x83,0x80,0x85,0x86,0x8f,0x8c,0x89,0x8a,
    0xab,0xa8,0xad,0xae,0xa7,0xa4,0xa1,0xa2,0xb3,0xb0,0xb5,0xb6,0xbf,0xbc,0xb9,0xba,
    0xfb,0xf8,0xfd,0xfe,0xf7,0xf4,0xf1,0xf2,0xe3,0xe0,0xe5,0xe6,0xef,0xec,0xe9,0xea,
    0xcb,0xc8,0xcd,0xce,0xc7,0xc4,0xc1,0xc2,0xd3,0xd0,0xd5,0xd6,0xdf,0xdc,0xd9,0xda,
    0x5b,0x58,0x5d,0x5e,0x57,0x54,0x51,0x52,0x43,0x40,0x45,0x46,0x4f,0x4c,0x49,0x4a,
    0x6b,0x68,0x6d,0x6e,0x67,0x64,0x61,0x62,0x73,0x70,0x75,0x76,0x7f,0x7c,0x79,0x7a,
    0x3b,0x38,0x3d,0x3e,0x37,0x34,0x31,0x32,0x23,0x20,0x25,0x26,0x2f,0x2c,0x29,0x2a,
    0x0b,0x08,0x0d,0x0e,0x07,0x04,0x01,0x02,0x13,0x10,0x15,0x16,0x1f,0x1c,0x19,0x1a};

int Mul9[256] = {
    0x00,0x09,0x12,0x1b,0x24,0x2d,0x36,0x3f,0x48,0x41,0x5a,0x53,0x6c,0x65,0x7e,0x77,
    0x90,0x99,0x82,0x8b,0xb4,0xbd,0xa6,0xaf,0xd8,0xd1,0xca,0xc3,0xfc,0xf5,0xee,0xe7,
    0x3b,0x32,0x29,0x20,0x1f,0x16,0x0d,0x04,0x73,0x7a,0x61,0x68,0x57,0x5e,0x45,0x4c,
    0xab,0xa2,0xb9,0xb0,0x8f,0x86,0x9d,0x94,0xe3,0xea,0xf1,0xf8,0xc7,0xce,0xd5,0xdc,
    0x76,0x7f,0x64,0x6d,0x52,0x5b,0x40,0x49,0x3e,0x37,0x2c,0x25,0x1a,0x13,0x08,0x01,
    0xe6,0xef,0xf4,0xfd,0xc2,0xcb,0xd0,0xd9,0xae,0xa7,0xbc,0xb5,0x8a,0x83,0x98,0x91,
    0x4d,0x44,0x5f,0x56,0x69,0x60,0x7b,0x72,0x05,0x0c,0x17,0x1e,0x21,0x28,0x33,0x3a,
    0xdd,0xd4,0xcf,0xc6,0xf9,0xf0,0xeb,0xe2,0x95,0x9c,0x87,0x8e,0xb1,0xb8,0xa3,0xaa,
    0xec,0xe5,0xfe,0xf7,0xc8,0xc1,0xda,0xd3,0xa4,0xad,0xb6,0xbf,0x80,0x89,0x92,0x9b,
    0x7c,0x75,0x6e,0x67,0x58,0x51,0x4a,0x43,0x34,0x3d,0x26,0x2f,0x10,0x19,0x02,0x0b,
    0xd7,0xde,0xc5,0xcc,0xf3,0xfa,0xe1,0xe8,0x9f,0x96,0x8d,0x84,0xbb,0xb2,0xa9,0xa0,
    0x47,0x4e,0x55,0x5c,0x63,0x6a,0x71,0x78,0x0f,0x06,0x1d,0x14,0x2b,0x22,0x39,0x30,
    0x9a,0x93,0x88,0x81,0xbe,0xb7,0xac,0xa5,0xd2,0xdb,0xc0,0xc9,0xf6,0xff,0xe4,0xed,
    0x0a,0x03,0x18,0x11,0x2e,0x27,0x3c,0x35,0x42,0x4b,0x50,0x59,0x66,0x6f,0x74,0x7d,
    0xa1,0xa8,0xb3,0xba,0x85,0x8c,0x97,0x9e,0xe9,0xe0,0xfb,0xf2,0xcd,0xc4,0xdf,0xd6,
    0x31,0x38,0x23,0x2a,0x15,0x1c,0x07,0x0e,0x79,0x70,0x6b,0x62,0x5d,0x54,0x4f,0x46};


int MulB[256] = {
    0x00,0x0b,0x16,0x1d,0x2c,0x27,0x3a,0x31,0x58,0x53,0x4e,0x45,0x74,0x7f,0x62,0x69,
    0xb0,0xbb,0xa6,0xad,0x9c,0x97,0x8a,0x81,0xe8,0xe3,0xfe,0xf5,0xc4,0xcf,0xd2,0xd9,
    0x7b,0x70,0x6d,0x66,0x57,0x5c,0x41,0x4a,0x23,0x28,0x35,0x3e,0x0f,0x04,0x19,0x12,
    0xcb,0xc0,0xdd,0xd6,0xe7,0xec,0xf1,0xfa,0x93,0x98,0x85,0x8e,0xbf,0xb4,0xa9,0xa2,
    0xf6,0xfd,0xe0,0xeb,0xda,0xd1,0xcc,0xc7,0xae,0xa5,0xb8,0xb3,0x82,0x89,0x94,0x9f,
    0x46,0x4d,0x50,0x5b,0x6a,0x61,0x7c,0x77,0x1e,0x15,0x08,0x03,0x32,0x39,0x24,0x2f,
    0x8d,0x86,0x9b,0x90,0xa1,0xaa,0xb7,0xbc,0xd5,0xde,0xc3,0xc8,0xf9,0xf2,0xef,0xe4,
    0x3d,0x36,0x2b,0x20,0x11,0x1a,0x07,0x0c,0x65,0x6e,0x73,0x78,0x49,0x42,0x5f,0x54,
    0xf7,0xfc,0xe1,0xea,0xdb,0xd0,0xcd,0xc6,0xaf,0xa4,0xb9,0xb2,0x83,0x88,0x95,0x9e,
    0x47,0x4c,0x51,0x5a,0x6b,0x60,0x7d,0x76,0x1f,0x14,0x09,0x02,0x33,0x38,0x25,0x2e,
    0x8c,0x87,0x9a,0x91,0xa0,0xab,0xb6,0xbd,0xd4,0xdf,0xc2,0xc9,0xf8,0xf3,0xee,0xe5,
    0x3c,0x37,0x2a,0x21,0x10,0x1b,0x06,0x0d,0x64,0x6f,0x72,0x79,0x48,0x43,0x5e,0x55,
    0x01,0x0a,0x17,0x1c,0x2d,0x26,0x3b,0x30,0x59,0x52,0x4f,0x44,0x75,0x7e,0x63,0x68,
    0xb1,0xba,0xa7,0xac,0x9d,0x96,0x8b,0x80,0xe9,0xe2,0xff,0xf4,0xc5,0xce,0xd3,0xd8,
    0x7a,0x71,0x6c,0x67,0x56,0x5d,0x40,0x4b,0x22,0x29,0x34,0x3f,0x0e,0x05,0x18,0x13,
    0xca,0xc1,0xdc,0xd7,0xe6,0xed,0xf0,0xfb,0x92,0x99,0x84,0x8f,0xbe,0xb5,0xa8,0xa3};

int MulD[256] = {
    0x00,0x0d,0x1a,0x17,0x34,0x39,0x2e,0x23,0x68,0x65,0x72,0x7f,0x5c,0x51,0x46,0x4b,
    0xd0,0xdd,0xca,0xc7,0xe4,0xe9,0xfe,0xf3,0xb8,0xb5,0xa2,0xaf,0x8c,0x81,0x96,0x9b,
    0xbb,0xb6,0xa1,0xac,0x8f,0x82,0x95,0x98,0xd3,0xde,0xc9,0xc4,0xe7,0xea,0xfd,0xf0,
    0x6b,0x66,0x71,0x7c,0x5f,0x52,0x45,0x48,0x03,0x0e,0x19,0x14,0x37,0x3a,0x2d,0x20,
    0x6d,0x60,0x77,0x7a,0x59,0x54,0x43,0x4e,0x05,0x08,0x1f,0x12,0x31,0x3c,0x2b,0x26,
    0xbd,0xb0,0xa7,0xaa,0x89,0x84,0x93,0x9e,0xd5,0xd8,0xcf,0xc2,0xe1,0xec,0xfb,0xf6,
    0xd6,0xdb,0xcc,0xc1,0xe2,0xef,0xf8,0xf5,0xbe,0xb3,0xa4,0xa9,0x8a,0x87,0x90,0x9d,
    0x06,0x0b,0x1c,0x11,0x32,0x3f,0x28,0x25,0x6e,0x63,0x74,0x79,0x5a,0x57,0x40,0x4d,
    0xda,0xd7,0xc0,0xcd,0xee,0xe3,0xf4,0xf9,0xb2,0xbf,0xa8,0xa5,0x86,0x8b,0x9c,0x91,
    0x0a,0x07,0x10,0x1d,0x3e,0x33,0x24,0x29,0x62,0x6f,0x78,0x75,0x56,0x5b,0x4c,0x41,
    0x61,0x6c,0x7b,0x76,0x55,0x58,0x4f,0x42,0x09,0x04,0x13,0x1e,0x3d,0x30,0x27,0x2a,
    0xb1,0xbc,0xab,0xa6,0x85,0x88,0x9f,0x92,0xd9,0xd4,0xc3,0xce,0xed,0xe0,0xf7,0xfa,
    0xb7,0xba,0xad,0xa0,0x83,0x8e,0x99,0x94,0xdf,0xd2,0xc5,0xc8,0xeb,0xe6,0xf1,0xfc,
    0x67,0x6a,0x7d,0x70,0x53,0x5e,0x49,0x44,0x0f,0x02,0x15,0x18,0x3b,0x36,0x21,0x2c,
    0x0c,0x01,0x16,0x1b,0x38,0x35,0x22,0x2f,0x64,0x69,0x7e,0x73,0x50,0x5d,0x4a,0x47,
    0xdc,0xd1,0xc6,0xcb,0xe8,0xe5,0xf2,0xff,0xb4,0xb9,0xae,0xa3,0x80,0x8d,0x9a,0x97};

int MulE[256] = {
    0x00,0x0e,0x1c,0x12,0x38,0x36,0x24,0x2a,0x70,0x7e,0x6c,0x62,0x48,0x46,0x54,0x5a,
    0xe0,0xee,0xfc,0xf2,0xd8,0xd6,0xc4,0xca,0x90,0x9e,0x8c,0x82,0xa8,0xa6,0xb4,0xba,
    0xdb,0xd5,0xc7,0xc9,0xe3,0xed,0xff,0xf1,0xab,0xa5,0xb7,0xb9,0x93,0x9d,0x8f,0x81,
    0x3b,0x35,0x27,0x29,0x03,0x0d,0x1f,0x11,0x4b,0x45,0x57,0x59,0x73,0x7d,0x6f,0x61,
    0xad,0xa3,0xb1,0xbf,0x95,0x9b,0x89,0x87,0xdd,0xd3,0xc1,0xcf,0xe5,0xeb,0xf9,0xf7,
    0x4d,0x43,0x51,0x5f,0x75,0x7b,0x69,0x67,0x3d,0x33,0x21,0x2f,0x05,0x0b,0x19,0x17,
    0x76,0x78,0x6a,0x64,0x4e,0x40,0x52,0x5c,0x06,0x08,0x1a,0x14,0x3e,0x30,0x22,0x2c,
    0x96,0x98,0x8a,0x84,0xae,0xa0,0xb2,0xbc,0xe6,0xe8,0xfa,0xf4,0xde,0xd0,0xc2,0xcc,
    0x41,0x4f,0x5d,0x53,0x79,0x77,0x65,0x6b,0x31,0x3f,0x2d,0x23,0x09,0x07,0x15,0x1b,
    0xa1,0xaf,0xbd,0xb3,0x99,0x97,0x85,0x8b,0xd1,0xdf,0xcd,0xc3,0xe9,0xe7,0xf5,0xfb,
    0x9a,0x94,0x86,0x88,0xa2,0xac,0xbe,0xb0,0xea,0xe4,0xf6,0xf8,0xd2,0xdc,0xce,0xc0,
    0x7a,0x74,0x66,0x68,0x42,0x4c,0x5e,0x50,0x0a,0x04,0x16,0x18,0x32,0x3c,0x2e,0x20,
    0xec,0xe2,0xf0,0xfe,0xd4,0xda,0xc8,0xc6,0x9c,0x92,0x80,0x8e,0xa4,0xaa,0xb8,0xb6,
    0x0c,0x02,0x10,0x1e,0x34,0x3a,0x28,0x26,0x7c,0x72,0x60,0x6e,0x44,0x4a,0x58,0x56,
    0x37,0x39,0x2b,0x25,0x0f,0x01,0x13,0x1d,0x47,0x49,0x5b,0x55,0x7f,0x71,0x63,0x6d,
    0xd7,0xd9,0xcb,0xc5,0xef,0xe1,0xf3,0xfd,0xa7,0xa9,0xbb,0xb5,0x9f,0x91,0x83,0x8d};

/*---------- Fonctions interaction avec l'utilisateur -----------------------*/

// Connaitre la taille d'un fichier de texte
int fsize(const char * fname)
{
    // La taille du fichier, si elle a pu etre calculee, est retournee dans *ptr
    int ptr;
    FILE * f;
    
    f = fopen(fname, "r");   
    if (f != NULL)
    {
        fseek(f, 0, SEEK_END); // aller a la fin du fichier
        ptr = ftell(f); // lire l'offset de la position courante par rapport au debut du fichier
        fclose(f);
        return ptr;
    }
    
    return 0; //En cas d'impossibilit de lecture du fichier, on retourne 0.
}

// Ouverture d'un fichier contenue texte (*.txt)
void ouverturefichier(const char * nomfichier,unsigned char *chaine){ //On ecrit ce qu'il y a dans fichier dans chaine.
    int i,j=0;
    int taille=fsize(nomfichier);
    unsigned char chaine1[taille+1];
   /* printf("%d",taille);*/
    // Recuperer ce qu'il y a dans le fichier txt
    FILE* fichier1=fopen(nomfichier,"r");  //lecture seule (ecriture : w)  

if (nomfichier != NULL)
    {             
         for(i=0;i<taille;i++){ 
            chaine1[i]=fgetc(fichier1);
         }
         for(i=0;i<taille;i++){
            if(chaine1[i]=='\\' && chaine1[i+1]=='n'){
               chaine[j]='\n';
               i++;
            }
            else chaine[j]=chaine1[i];
            
            j++;
         }
    } 
    fclose(fichier1);
    
    //Affichage du fichier ouvert et du nombre de caractres :
   /* printf("Message/clef :\n");            
    for(i=0;i<taille;i++){
       printf("%c",chaine[i]);                    
    }
    printf("\nNombre de caracteres : %d",taille);
    */
}

// Ouverture d'un fichier contenue en HEXA (*.txt)
void ouverturefichierHEXA(const char * nomfichier,unsigned char *chaine){ //On ecrit ce qu'il y a dans fichier dans chaine.
    int i,a,b;
    int taille=fsize(nomfichier);
   // printf("%d",taille);
    // Recuperer ce qu'il y a dans le fichier txt
    FILE* fichier1=fopen(nomfichier,"r");  //lecture seule (ecriture : w)  

if (nomfichier != NULL)
    {             
         for(i=0;i<taille/2;i++){ 
          a=fgetc(fichier1);
          b=fgetc(fichier1);

            if(a>='0' && a<='9') a=a-'0';
            else a=a-'a'+10;

            if(b>='0' && b<='9') b=b-'0';
            else b=b-'a'+10;
      
            chaine[i]=a*16+b;          
         }
    } 
    fclose(fichier1);
    
}


/* Ouverture d'une clef contenue dans un fichier texte
   On ecrit ce qu'il y a dans fichier dans chaine.*/
void ouvertureclef(const char * nomfichier,unsigned char *chaine){
    int i;
    int taille=fsize(nomfichier);
    
    // Recuperer ce qu'il y a dans le fichier txt
    FILE* fichier1=fopen(nomfichier,"r");  //lecture seule (ecriture : w)  

if (nomfichier != NULL && taille==16){    
            for(i=0;i<taille;i++){ 
            chaine[i]=fgetc(fichier1);
          }
    fclose(fichier1);
    
    //Affichage du fichier ouvert et du nombre de caractres :
   /* printf("Message/clef :\n");            
    for(i=0;i<taille;i++){
       printf("%c",chaine[i]);                    
    }
    printf("\nNombre de caracteres : %d",taille);
    */
}
}


/* Ouverture d'une clef contenue dans un fichier texte
   On ecrit ce qu'il y a dans fichier dans chaine.*/
void ouvertureclefHEXA(const char * nomfichier,unsigned char *chaine){
    int i;
    char a,b;
    int taille=fsize(nomfichier);
    
    // Recuperer ce qu'il y a dans le fichier txt
    FILE* fichier1=fopen(nomfichier,"r");  //lecture seule (ecriture : w)  

if (nomfichier != NULL && taille==16){    

if (nomfichier != NULL)
    {             
         for(i=0;i<taille/2;i++){ 
          a=fgetc(fichier1);
          b=fgetc(fichier1);

            if(a>='0' && a<='9') a=a-'0';
            else a=a-'a'+10;

            if(b>='0' && b<='9') b=b-'0';
            else b=b-'a'+10;
      
            chaine[i]=a*16+b;          
         }
    } 
          }
    fclose(fichier1);
    
    //Affichage du fichier ouvert et du nombre de caractres :
   /* printf("Message/clef :\n");            
    for(i=0;i<taille;i++){
       printf("%c",chaine[i]);                    
    }
    printf("\nNombre de caracteres : %d",taille);
    */
}


// Ecrire des donnes dans un fichier texte en ASCII
void editerfichier(const char * nomfichier,unsigned char *chaine,int taillechaine){ //On ecrit ce qu'il y a dans fichier dans chaine.
    int i;
    //printf("%d",taille);
    // Recuperer ce qu'il y a dans le fichier txt
    FILE* fichier1=fopen(nomfichier,"w");  //mode ecriture

    if(nomfichier != NULL){    
      for(i=0;i<taillechaine;i++) fprintf(fichier1,"%c",chaine[i]);
    }
    else printf("\nLe fichier n'existe pas ...\n");
    fclose(fichier1);
    
    //Affichage du fichier ouvert et du nombre de caractres :
    /*printf("Message :\n");            
    for(i=0;i<taillechaine;i++){
       printf("%c",chaine[i]);                    
    }
    printf("\nNombre de caracteres : %d",taillechaine);
   */ 
}

// Ecrire des donnes dans un fichier texte en HEXA
void editerfichierHEXA(const char * nomfichier,unsigned char *chaine,int taillechaine){ //On ecrit ce qu'il y a dans fichier dans chaine.
    int i;
    //printf("%d",taille);
    // Recuperer ce qu'il y a dans le fichier txt
    FILE* fichier1=fopen(nomfichier,"w");  //mode ecriture

    if(nomfichier != NULL){    
      for(i=0;i<taillechaine;i++) fprintf(fichier1,"%02.2x",chaine[i]);
    }
    else printf("\nLe fichier n'existe pas ...\n");
    fclose(fichier1);
    
    //Affichage du fichier ouvert et du nombre de caractres :
    /*printf("Message :\n");            
    for(i=0;i<taillechaine;i++){
       printf("%c",chaine[i]);                    
    }
    printf("\nNombre de caracteres : %d",taillechaine);
    */
}


// La fonction affichage en HEXA d'un message 
void affichagehexa(unsigned char *in) //en hxa
{
  int j;
  for(j=0;j<TAILLEMESSAGE;j++){
     printf("%02.2x ",in[j]);// Affichage en hxa (pour verifier avec la norme) 
  }
  printf("\n");
}

// Affichage en ASCII d'un message
void affichage(unsigned char *in){
  int j; // N'afficher que les 32 premiers caractres au maximum
  if(TAILLEMESSAGE>32){
     for(j=0;j<32;j++){
       printf("%c",in[j]);
     }
  }
  else for(j=0;j<TAILLEMESSAGE;j++){
          printf("%c",in[j]);
       }
       
  printf("\n");
}

// Affichage de la clef en ASCII (16 char)
void affichageclef(unsigned char *in){
  int j;
  for(j=0;j<16;j++){
     printf("%c",in[j]);
  }
  printf("\n");
}

// La fonction affichage_binaire affiche un bloc de message en binaire
void affichage_binaire(unsigned char *in)
{
  int i,j,k;

  for(i=0;i<4;i++){
      for(j=0;j<4;j++){
	    printf("%u%u%u%u",(in[4*j+i] >>7) & 1,(in[4*j+i] >>6)& 1,(in[4*j+i] >>5)& 1,(in[4*j+i] >> 4) & 1);
	    printf("%u%u%u%u  ",(in[4*j+i] >>3) & 1,(in[4*j+i] >>2)& 1,(in[4*j+i] >>1)& 1,in[4*j+i] & 1);
      }
      printf("\n");
  }
  printf("\n");
}

//Creer un nombe alatoire entre iMin et iMax
int Random (int iMin, int iMax){
    return (iMin + (rand () % (iMax-iMin+1)));
} 

// Affichage du menu principal
int affichageMenu(){
     int choixMenu;
     
     printf("---- AES 128 bits par Quentin Comte-Gaz et Adrien Lillo ----\n-                                                          -\n");
     printf("-1.Creation d'une clef aleatoire                           -\n");
     printf("-2.Cryptage du fichier texteclair.txt (clef : clef.txt)    -\n");
     printf("-3.Decryptage du fichier textecrypte.txt (clef : clef.txt) -\n");     
     printf("-4.Faire le test du document fips197 definissant l'AES     -\n");
     printf("-5.Quitter le programme                                    -\n");
     printf("------------------------------------------------------------\n");
     printf("\nTapez le numero correspondant a votre choix :\n");
     scanf("%d", &choixMenu);
     return choixMenu;
}

/*----------------------Fonctions propres  l'AES--------------------------*/



// La fonction Add fait un XOR bit a bit avec deux blocs de 16 octets
void Add(unsigned char out[], unsigned char in1[],unsigned char in2[])
{
	int i,j;
	for(i=0;i<16;i++){
	  out[i] =in1[i] ^in2[i];		
	}// ^ == xor binaire
}

// Copie la valeur les octets du tableau in dans le tableau out
void Affecte(unsigned char out[], unsigned char in[])
{
  int i,j;
  for(i=0;i<16;i++){
      out[i] =in[i];		
  }
}

/* La fonction KeyExpansion produit les 4(Nr+1) clef de rondes, 
o Nr est le nombre de clef de rondes*/
//Cf p20 du fichier de norme pour le pseudo code
void KeyExpansion( unsigned char *RoundKey, unsigned char Key[], int Nr)
{
  int i,j;
  unsigned char temp[4],k;
	
  // La premire clef de ronde est la clef elle mme.
  for(j=0;j<Nk;j++){
      RoundKey[j*4]=Key[j*4];
      RoundKey[j*4+1]=Key[j*4+1];
      RoundKey[j*4+2]=Key[j*4+2];
      RoundKey[j*4+3]=Key[j*4+3];
  }

  /* On cr maintenant les autres clefs de rondes  l'aide 
  des clefs de ronde precedentes).*/
  while (j < (4 * (Nr+1))){
      for(i=0;i<4;i++){
	     temp[i]=RoundKey[(j-1) * 4 + i];
	  }
      if (j % Nk == 0){
	  /* Cette fonction ("RotWord") fait une rotation de 4 char :
         [a0,a1,a2,a3] devient [a1,a2,a3,a0] : */
	  {
	    k = temp[0];
	    temp[0] = temp[1];
	    temp[1] = temp[2];
	    temp[2] = temp[3];
	    temp[3] = k;
	  }

	  /* Cette fonction (SubWord) applique la S-Box aux 4 char 
      pour produire un mot en sortie*/
	  {
	    temp[0]=sbox[temp[0]];
	    temp[1]=sbox[temp[1]];
	    temp[2]=sbox[temp[2]];
	    temp[3]=sbox[temp[3]];
	  }

	  temp[0] =  temp[0] ^ Rcon[j/Nk];
	}
      else if ((Nk > 6) && ((j % Nk) == 4))
	{
 
	  {
	    temp[0]=sbox[temp[0]];
	    temp[1]=sbox[temp[1]];
	    temp[2]=sbox[temp[2]];
	    temp[3]=sbox[temp[3]];
	  }
	}
      // Il ne reste plus qu' creer les clefs de rondes : 
      RoundKey[j*4+0] = RoundKey[(j-Nk)*4+0] ^ temp[0];
      RoundKey[j*4+1] = RoundKey[(j-Nk)*4+1] ^ temp[1];
      RoundKey[j*4+2] = RoundKey[(j-Nk)*4+2] ^ temp[2];
      RoundKey[j*4+3] = RoundKey[(j-Nk)*4+3] ^ temp[3];
      j++;
    }
}

// La fonction AddRoundKey ajoute la clef de ronde  *in
void AddRoundKey(unsigned char *out, unsigned char *in,unsigned char RoundKey[], int round) 
{        
	int i,j;
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			out[4*j+i] =in[4*j+i]^ RoundKey[round * 16 + j * 4 + i];  
		}
	}
}

// La fonction  SubBytes substitue les valeurs des 16 octets du bloc de message
void SubBytes(unsigned char *out, unsigned char *in)
{
	int i,j;
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
		  out[4*j+i] = (unsigned char) sbox[in[4*j+i]];

		}
	}
}

// La fonction  SubBytes substitue les valeurs des 16 octets du bloc de message
void InvSubBytes(unsigned char *out, unsigned char *in)
{
	int i,j;
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
		  out[4*j+i] = (unsigned char) sboxinv[in[4*j+i]];

		}
	}
}



//  La fonction ShiftRows() decale cycliquement les lignes du bloc message
// de 0,1,2,3 sur la la gauche pour les 1ere, 2eme, 3eme, 4eme ligne respectivement
void ShiftRows(unsigned char *out, unsigned char *in)
{
	int j;
	// la premiere ligne est inchangee
	for(j=0;j<4;j++)
	  {
	    out[4*j]=in[4*j];
	  }
	// deuxieme ligne on decalage de 1 sur la gauche 
	for(j=0;j<3;j++)
	  {
	    out[4*j+1]=in[4*(j+1)+1];
	  }
	out[4*3+1]=in[1];
	// troisime ligne on decalage de 2 sur la gauche 
	for(j=0;j<2;j++)
	  {
	    out[4*j+2]=in[4*(j+2)+2];
	    out[4*(j+2)+2]=in[4*(j)+2];
	  }
	out[4*3+1]=in[1];
	// quatrieme ligne on decalage de 3 sur la droite
	for(j=1;j<4;j++)
	  {
	    out[4*j+3]=in[4*(j-1)+3];
	  }
	out[3]=in[4*3+3];
}



//  La fonction ShiftRows() decale cycliquement les lignes du bloc message
// de 0,1,2,3 sur la la gauche pour les 1er, 2eme, 3eme, 4eme ligne respectivement
void InvShiftRows(unsigned char *out, unsigned char *in)
{
	int j;
	// la premiere ligne est inchangee
	for(j=0;j<4;j++)
	  {
	    out[4*j]=in[4*j];
	  }
	// deuxieme ligne on decale de 1 sur la gauche 
	for(j=1;j<4;j++)
	  {
	    out[4*j+1]=in[4*(j-1)+1];
	  }
	out[1]=in[4*3+1];
	// troisieme ligne on decale de 2 sur la gauche 
	for(j=0;j<2;j++)
	  {
	    out[4*j+2]=in[4*(j+2)+2];
	    out[4*(j+2)+2]=in[4*(j)+2];
	  }
	//out[4*3+1]=in[1];
	// quatrieme ligne on decale de 1 sur la droite
	for(j=0;j<3;j++)
	  {
	    out[4*j+3]=in[4*(j+1)+3];
	  }
	out[4*3+3]=in[3];
}


// La fonction MixColumns melange les colonnes du bloc de message
void MixColumns(unsigned char *out, unsigned char *in)
{
  int i,j;
  unsigned char Tmp,Tm,t;
  
  for(j=0;j<4;j++)
    {	
      
      Tmp = in[0+j*4] ^ in[1+j*4] ^ in[2+j*4] ^ in[3+j*4] ;
      Tm = in[0+j*4] ^ in[1+j*4] ; 
      Tm = xtime(Tm); 
      out[0+j*4] = Tm ^ Tmp ^  in[0+j*4];
      Tm = in[1+j*4] ^ in[2+j*4] ; 
      Tm = xtime(Tm); 
      out[1+j*4] = Tm ^ Tmp ^ in[1+j*4];
      Tm = in[2+j*4] ^ in[3+j*4] ; 
      Tm = xtime(Tm); 
      out[2+j*4] = Tm ^ Tmp ^ in[2+j*4];
      Tm = in[3+j*4] ^ in[0+j*4]; 
      Tm = xtime(Tm);
      out[3+j*4] = Tm ^ Tmp ^ in[3+j*4];
    }
}


// La fonction MixColumns melange les colonnes du bloc de message
void InvMixColumns(unsigned char *out, unsigned char *in)
{
  int i,j;
  unsigned char Tmp,Tm,t;

  // Reordering the equations we get:
  // y_0  = 0e * x_0 + 0b * x_1 + 0d * x_2 + 09 * x_3 	 
  // y_1  = 0e * x_1 + 0b * x_2 + 0d * x_3 + 09 * x_0 	 
  // y_2  = 0e * x_2 + 0b * x_3 + 0d * x_0 + 09 * x_1 	 
  // y_3  = 0e * x_3 + 0b * x_0 + 0d * x_1 + 09 * x_2.

  for(j=0;j<4;j++)
    {	
      out[0+j*4] = MulE[in[0+j*4]] ^ MulB[in[1+j*4]] ^ MulD[in[2+j*4]] ^ Mul9[in[3+j*4]];
      out[1+j*4] = MulE[in[1+j*4]] ^ MulB[in[2+j*4]] ^ MulD[in[3+j*4]] ^ Mul9[in[0+j*4]];
      out[2+j*4] = MulE[in[2+j*4]] ^ MulB[in[3+j*4]] ^ MulD[in[0+j*4]] ^ Mul9[in[1+j*4]];
      out[3+j*4] = MulE[in[3+j*4]] ^ MulB[in[0+j*4]] ^ MulD[in[1+j*4]] ^ Mul9[in[2+j*4]];
    }

}

void Cut(unsigned char *M16, unsigned char M[], int n){
         int i=0;
         for(i=0;i<16;i++){
         M16[i]=M[16*n+i];                  
         }     
}



// La fonction cipher sert a chiffrer un bloc message de 16 char
void Cipher(unsigned char *C, unsigned char M[], unsigned char Key[], int Nr)
{
    if(TEST==1) printf("\nCIPHER (ENCRYPT):\n");  
	int i,j,round=0;
	unsigned char T[16],U[16],V[16],W[16];
	unsigned char RoundKey[240];
   //On cherche le nombre de bloc du message

    //On refait le chiffrage pour chaque bloc du message
	
    //On fabrique les clefs de ronde
	KeyExpansion( RoundKey, Key, 10);
 
   
	//On copide le messge dans le bloc d'etat 
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			W[4*j+i] = M[4*j + i];
		}
	}
	
    if(TEST==1){	
                printf("round[ 0].input ");
                affichagehexa(W);
    }
	// AddKey de la premiere clef
	AddRoundKey(T,W,RoundKey,0); 
	// Il y a Nr rondes.
	// Les Nr-1 premieres rondes sont identiques
	// Ces Nr-1 rondes sont effectuées dans la boucle for ci-dessous
	for(round=1;round<Nr;round++)
	{
     if(TEST==1){	
                printf("round[ %d].start ",round);
                affichagehexa(T);
     }
                              
	SubBytes(U,T);
     if(TEST==1){	
                printf("round[ %d].s_box ",round);
                affichagehexa(U);
     } 
    
 	ShiftRows(V,U);
     if(TEST==1){	
                printf("round[ %d].s_row ",round);
                affichagehexa(V);
     }
      
 	MixColumns(W,V);
     if(TEST==1){	
                printf("round[ %d].m_col ",round);           
                affichagehexa(W);
     }
      	  
	  AddRoundKey(T,W,RoundKey,round);
	  
	}	
	// Derniere ronde
	// La fonction MixColumns n'est pas dans la derniere ronde
     if(TEST==1){	
                printf("round[%d].start ",round);
                affichagehexa(T);
     }
    
	SubBytes(U,T);
     if(TEST==1){	
                printf("round[%d].s_box ",round);
                affichagehexa(U);
     }	
    
	ShiftRows(V,U);
     if(TEST==1){	
                printf("round[%d].s_row ",round);
                affichagehexa(V);
     }
    
	AddRoundKey(W,V,RoundKey,Nr);	
    
     if(TEST==1){	
                printf("round[%d].output",round);
                affichagehexa(W);
     }

	// Le processus de chiffremen est fini
	// On copie le bloc d'etat du message dans le bloc de message chiffre
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			C[j*4+i]=W[4*j+i];
		}
	}
  
}

 /* Ne marche pas trs bien*//*    
int count(unsigned char *tab){
     int i;
     for (i=0;tab[i];i++);
     return i;

}*/

// Problme avec les sauts de lignes
// CipherComplet permet de chiffrer un message de taille quelconque grace  Cipher
void CipherC(unsigned char *C1, unsigned char M[], unsigned char Key[], int Nr){
     int i,j,iteration=0,nBloc=0;
     unsigned char C[16];
     unsigned char M16[16];

    if(TAILLEMESSAGE>16){
        if(TAILLEMESSAGE%16==0) iteration = TAILLEMESSAGE/16-1;
        else iteration = TAILLEMESSAGE/16;            
    }
   // printf("\n\nReste : %d\nTaille message : %d\nIterations : %d\n\n",TAILLEMESSAGE%16,TAILLEMESSAGE,iteration+1);      
        
    for(i=0;i<iteration+1;i++){
         Cut(M16,M,nBloc);
         Cipher(C,M16,Key,Nr);
         
         for(j=0;j<16;j++){
              C1[nBloc*16+j]=C[j];
         }
         nBloc++;
    }
}



// Fonction de dechiffrement de 16 char
void Decipher(unsigned char *out, unsigned char in[], unsigned char Key[], int Nr)
{
    if(TEST==1) printf("\nINVERSE CIPHER (DECRYPT):\n");
    
	int i,j,round=0;
	unsigned char T[16],U[16], V[16],W[16];
	unsigned char RoundKey[240];


	//On fabrique les clefs de ronde
	KeyExpansion( RoundKey, Key, 10);

	//On copide le messge dans le bloc d'etat 
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			W[4*j+i] = in[4*j + i];
		}
	}
	
	if(TEST==1){
               printf("round[ 0].iinput ");
               affichagehexa(W);
    }
	// On "pele" derniere ronde
	// La fonction MixColumns n'est pas dans la derniere ronde
	AddRoundKey(V,W,RoundKey,Nr);	
    
	InvShiftRows(U,V);
 
	InvSubBytes(T,U);


	// Il y a Nr rondes.
	// Les Nr-1 premieres rondes sont identiques
	// Ces Nr-1 rondes sont dechiffrees dans la boucle for ci-dessous
	for(round=Nr-1;round>0;round--)
	{   
	  AddRoundKey(W,T,RoundKey,round);
	  InvMixColumns(V,W);	  
	  InvShiftRows(U,V);	  
	  InvSubBytes(T,U);
	}	

	// AddKey de la premiere clef
	AddRoundKey(W,T,RoundKey,0); 
	
    if(TEST==1){
               printf("round[%d].ioutput",(10-round));
               affichagehexa(W);
    }
    
	// Le processus de dechiffremen est fini
	// On copie le bloc d'etat du message dans le bloc de message clair (out)
	for(i=0;i<4;i++)
	{
		for(j=0;j<4;j++)
		{
			out[j*4+i]=W[4*j+i];
		}
	}
}

// DeCipherComplet permet de dchiffrer un message de taille quelconque grace  DeCipher
void DecipherC(unsigned char *out, unsigned char in[], unsigned char Key[], int Nr){
     int i,j,iteration=0,nBloc=0;
    unsigned char out1[16];
    unsigned char in16[16];
   
    if(TAILLEMESSAGE>16){
        if(TAILLEMESSAGE%16==0) iteration = TAILLEMESSAGE/16-1;
        else iteration = TAILLEMESSAGE/16;               
    }
    // printf("\n\nReste : %d\nTaille message : %d\nIterations : %d\n\n",TAILLEMESSAGE%16,TAILLEMESSAGE,iteration+1);      
    
    for(i=0;i<iteration+1;i++){
         Cut(in16,in,nBloc);
         Decipher(out1,in16,Key,Nr);
         
         for(j=0;j<16;j++){
              out[nBloc*16+j]=out1[j];
         }
         nBloc++;
    }
}

/*---------------- Fonction principale ----------------------------------*/

int main (){
    int choixMenu=0,j,i;
    
    for(j=0;j<16;j++) {if(fsize("texteclair.txt")%16!=0);
                      else TAILLEMESSAGE++;}
    
    unsigned char M[fsize("texteclair.txt")]; // La taille du message crypt et du message dcrypt est la meme (... elle doit etre modulo 16=0...  faire)
    unsigned char C[fsize("textecrypte.txt")];
    unsigned char CHEXA[fsize("textecrypte.txt")/2];   
    unsigned char Key[16]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f}; //Clef par defaut  
    FILE* fichier;  
    srand(time(NULL));

    
    while(choixMenu!=5){

    //Affichage de notre menu
    choixMenu = affichageMenu();   
   
     
    //Les differentes fonctions  effectuer
        switch(choixMenu){   
                     case 1:{
                          printf("\nNouvelle clef aleatoire (clef.txt) :\n");
                          // fonction qui permet de creer une fonction de creation de nombres aleatoire, afficher cette clef de 16 char et l'enregistrer dans le fichier clef.txt
                         int i,j;
                         char clef[16];
                          for(i=0;i<16;i++){
                          clef[i]=Random(0,255);                  
                          printf("%c",clef[i]);
                          }
                          printf("\n\n");
                          editerfichierHEXA("clef.txt",clef,16);
                          break;}
                     case 2:{
                          printf("\nCryptage (texteclair.txt+clef.txt-->textecrypte.txt) :\n"); // On crypte :
                          fichier=fopen("texteclair.txt","r");
                          TAILLEMESSAGE=fsize("texteclair.txt");
                          
                          
                          fclose(fichier);
                          ouverturefichier("texteclair.txt",M); 
                          
                          for(j=fsize("texteclair.txt");j<fsize("texteclair.txt")+16;j++) {if(M[j]=='\0') M[TAILLEMESSAGE-j]=' ';}// Rajouter des elements au message (espaces) pour que celui ci soit modulo16=0 (un nombre entier de blocs de 16 char)
                          
                          ouvertureclef("clef.txt",Key); 
                          printf("Notre clef de cryptage :\n");
                          affichageclef(Key);  
                          printf("Les 32 premiers caracteres du message a crypter :\n");
                          affichage(M);
                          CipherC(C, M, Key,10);
                          printf("Les 32 premiers caracteres du message crypte :\n");
                          affichage(C);
                          /*EDITER LE FICHIER SOUS FORME HEXA*/
                          editerfichierHEXA("textecrypte.txt",C,TAILLEMESSAGE); // enregistrer le texte crypt dans le fichier textecrypte.txt
                          printf("\n");
                          break;}
                     case 3:{
                          printf("\nDecryptage (textecrypte.txt+clef.txt-->texteclair.txt) :\n"); // On crypte :                          
                          fichier=fopen("textecrypte.txt","r");
                          TAILLEMESSAGE=fsize("textecrypte.txt")/2; 
                          fclose(fichier);
                          ouvertureclef("clef.txt",Key); 
                          /* OUVRIR LE TEXTE CHIFFRE SOUS FORME HEXA ET TRADUIRE EN ASCII*/
                          ouverturefichierHEXA("textecrypte.txt",CHEXA); 
                          printf("Notre clef de cryptage :\n");
                          affichageclef(Key);  
                          printf("Les 32 premiers caracteres du message a decrypter :\n");
                          affichage(CHEXA);                         
                          DecipherC(M,CHEXA,Key,10);
                          printf("Les 32 premiers caracteres du message decrypte :\n");
                          affichage(M);
                          editerfichier("texteclair.txt",M,TAILLEMESSAGE); // enregistrer le texte crypt dans le fichier texteclair.txt   
                          printf("\n");                                                                     
                          break;}
                     case 4:{
                          TAILLEMESSAGE=16;
                          TEST=1;
                          unsigned char Keytest[16]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f}; //clef de test
                          unsigned char Mtest[32]={0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77,0x88,0x99,0xaa,0xbb,0xcc,0xdd,0xee,0xff}; //'a','b','c','d','e','f','g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z','0','1','2','3','4','5'};
                          unsigned char Ctest[32];
                          CipherC(Ctest, Mtest, Keytest,10);
                          DecipherC(Mtest,Ctest,Keytest,10);
                          printf("\n");
                          TEST=0;
                          break;}                           
                     case 5:{
                          printf("\nArret du programme en cours ...\n\n");
                          return 0;
                          break;}
                     default:{
                             printf("Veuillez taper un NOMBRE ENTRE 1 ET 5 !\n\n");
                             break;}
        }
   /* 
    FILE* fichier=fopen("texteclair.txt","r");
    unsigned char M[fsize("texteclair.txt")]; // La taille du message crypt et du message dcrypt est la meme.
    unsigned char C[fsize("texteclair.txt")];  
    TAILLEMESSAGE=fsize("texteclair.txt"); 
    fclose(fichier);
    ouverturefichier("texteclair.txt",M);   */ 
    
    /*unsigned char M[TAILLEMESSAGE]={0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77,0x88,0x99,0xaa,0xbb,0xcc,0xdd,0xee,0xff};
    unsigned char C[TAILLEMESSAGE];*/
    /*unsigned char Key[16]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f}; //Clef par defaut
    ouvertureclef("clef.txt",Key); 
    printf("Notre clef de cryptage :\n");
    affichage(Key);
             
    printf("Notre message a crypter :\n");
     affichage(M);
     Cipher(C, M, Key,10);
     printf("Notre message crypte :\n");
     affichage(C);
     Decipher(C,C,Key,10);
     printf("Notre message decrypte :\n");
     affichage(C);
     */
     }

}

